(function () {
    let bookmarkList = document.getElementById('bookmark_list');
    let entryTemplate = document.getElementById('entry_template').cloneNode(true);
    let enableInline = undefined;
    chrome.storage.local.get(['enableInline'], result => {
        if (result.enableInline) {
            document.getElementById('toggle_inline').textContent = 'ON';
            document.getElementById('toggle_inline').style.color = 'green';
        } else {
            document.getElementById('toggle_inline').textContent = 'OFF';
            document.getElementById('toggle_inline').style.color = 'red';
        }
        enableInline = result.enableInline
    });
    document.getElementById('entry_template').remove();
    const on_icons = {
        "16": "images/16_on.png",
        "32": "images/32_on.png",
        "48": "images/48_on.png",
        "128": "images/128_on.png"
    };
    const off_icons = {
        "16": "images/16_off.png",
        "32": "images/32_off.png",
        "48": "images/48_off.png",
        "128": "images/128_off.png"
    };


    // Append a single bookmark to the DOM
    function appendBookmarkToDOM(entry) {
        let bookmarkElement = entryTemplate.cloneNode(true);
        let link = bookmarkElement.querySelector('a');
        link.setAttribute('href', entry.url);
        link.setAttribute('title', entry.url);
        link.textContent = entry.title;
        bookmarkElement.removeAttribute('id');
        bookmarkElement.setAttribute('data-id', entry.id);
        bookmarkList.appendChild(bookmarkElement);
    }

    // Remove a bookmark from the DOM
    function removeBookmarkFromDOM(bookmark_id) {
        bookmarkList.querySelector(`[data-id="${bookmark_id}"]`).remove();
    }

    // Refresh the list of bookmarks
    function reloadBookmarks() {
        document.getElementById('bookmark_list').innerHTML = '';
        chrome.storage.local.get(['bookmarks']).then(result => {
            if (result.bookmarks.length > 0) {
                result.bookmarks.forEach(entry => {
                    appendBookmarkToDOM(entry);
                });
            }
        });
    }

    // Add a new bookmark to storage
    function addBookmark(newEntry) {
        chrome.storage.local.get(['bookmarks']).then(result => {            
            let bookmarks = result.bookmarks;
            bookmarks.push(newEntry);
            chrome.storage.local.set({"bookmarks": bookmarks});
            // Add new bookmark to DOM
            appendBookmarkToDOM(newEntry);
        });
    }

    // Refresh bookmark list when user opens the pop-up page
    reloadBookmarks();

    // If user clicks anywhere within the bookmark list
    bookmarkList.addEventListener('click', e => {
        // If user clicked the bookmark link, open it in a new tab
        if (e.target.hasAttribute('data-bookmark-link')) {
            e.preventDefault();
            chrome.tabs.create({ url: e.target.href });
        // If user clicked delete icon, then delete the bookmark
        } else if (e.target.hasAttribute('data-delete-bookmark')) {
            // Get the bookmark ID
            let bookmarkID = e.target.closest('div[data-id]').getAttribute('data-id');
            // Retrieve bookmark data from storage
            chrome.storage.local.get(['bookmarks']).then(result => {
                let bookmarks = result.bookmarks;
                // Remove bookmark from array
                let arrayIndex = bookmarks.findIndex(bookmark => bookmark.id === bookmarkID);
                bookmarks.splice(arrayIndex, 1);
                // Remove bookmark from DOM
                removeBookmarkFromDOM(bookmarkID);
                // Push updated array of bookmarks to local storage
                chrome.storage.local.set({"bookmarks": bookmarks});
            });
        // If user clicked edit icon, then display editing fields
        } else if (e.target.hasAttribute('data-edit-bookmark')) {
            let bookmarkContainer = e.target.closest('div[data-id]');
            let newLabel = bookmarkContainer.querySelector('[data-new-bookmark-label]');
            
            // Get current auto-calculated width of bookmark container and set it to that as a static value.
            // This will ensure the width of the pop-up form doesn't change as you edit the bookmark
            let computedWidth = getComputedStyle(bookmarkContainer).width;
            bookmarkContainer.style.width = computedWidth;
            
            // Hide the bookmark read-only elements
            bookmarkContainer.querySelector('[data-read-only-container]').style.display = 'none';
            // Unhide the bookmark edit form elements
            bookmarkContainer.querySelector('[data-update-container]').style.display = 'flex';

            // Populate bookmark's current label to the input box and focus cursor into it
            newLabel.value = bookmarkContainer.querySelector('[data-bookmark-link').textContent;
            newLabel.focus();
        // If user clicks Cancel on bookmark edit form, close out the form
        } else if (e.target.hasAttribute('data-cancel-update')) {
            let bookmarkContainer = e.target.closest('div[data-id]');
            bookmarkContainer.querySelector('[data-update-container]').style.display = 'none';
            bookmarkContainer.querySelector('[data-read-only-container]').style.display = 'inline';
        // If user clicks Save on bookmark edit form, update bookmark's name and close form
        } else if (e.target.hasAttribute('data-update-bookmark')) {
            let bookmarkContainer = e.target.closest('div[data-id]');
            let userInputField = bookmarkContainer.querySelector('[data-new-bookmark-label]');
            let newBookmarkName = userInputField.value.trim();
            // Ensure new name isn't empty
            if (newBookmarkName === '') {
                userInputField.style.border = '2px solid red';
                setTimeout(() => {
                    userInputField.style.border = '';
                }, 750);
                return;
            }
            // Grab bookmarks from storage
            chrome.storage.local.get(['bookmarks']).then(result => {
                let bookmarks = result.bookmarks;
                let bookmarkID = bookmarkContainer.getAttribute('data-id');
                // Update the bookmark
                for (let i = 0; i < bookmarks.length; i++) {
                    if (bookmarks[i].id === bookmarkID) {
                        bookmarks[i].title = newBookmarkName;
                        break;
                    }
                }
                // Push updated bookmark list to storage
                chrome.storage.local.set({bookmarks: bookmarks});

                // Hide edit form
                bookmarkContainer.querySelector('[data-update-container]').style.display = 'none';

                // Update and display read-only bookmark container
                bookmarkContainer.querySelector('[data-bookmark-link]').textContent = newBookmarkName;
                bookmarkContainer.querySelector('[data-read-only-container]').style.display = 'inline';
            });
        }
    });

    // Event listener for when user clicks button to add a new bookmark
    document.getElementById('bookmark_page').addEventListener('click', () => {
        chrome.tabs.query({ active: true, currentWindow: true}, (tabs) => {
            let newEntry = {id: crypto.randomUUID(), title: tabs[0].title, url: tabs[0].url }
            addBookmark(newEntry);
        });
    });

    // Change appearance of the button used to toggle inline bookmark
    document.getElementById('toggle_inline').addEventListener('click', () => {
        if (enableInline) {
            enableInline = false;
            chrome.storage.local.set({ enableInline: false });
            document.getElementById('toggle_inline').textContent = 'OFF';
            document.getElementById('toggle_inline').style.color = 'red';
            chrome.action.setIcon({ path: off_icons });
        } else {
            enableInline = true;
            chrome.storage.local.set({ enableInline: true });
            document.getElementById('toggle_inline').textContent = 'ON';
            document.getElementById('toggle_inline').style.color = 'green';
            chrome.action.setIcon({ path: on_icons });
        }
            
    });

    // Display the info panel when user hovers over the label
    document.getElementById('toggle_inline_label').addEventListener('mouseenter', () => {
        document.getElementById('info_caption').style.display = 'block';
    });

    // Hide info panel when user moves mouse off the label
    document.getElementById('toggle_inline_label').addEventListener('mouseleave', () => {
        document.getElementById('info_caption').style.display = 'none';
    });
})();